<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Spatie\Activitylog\Traits\LogsActivity;

/**
 * Class Invoice
 *
 * @property int $id
 * @property string $invoice_number
 * @property string $invoice_paper_number
 * @property int|null $customer_id
 * @property string|null $discount_type
 * @property float|null $discount_amount
 * @property string $status
 * @property float $sub_total
 * @property float $total_amount_paid
 * @property float $total_profit
 * @property float $total_cost
 * @property float $vat
 * @property float $vat_amount
 * @property int|null $created_by
 * @property int|null $last_updated_by
 * @property int|null $voided_by
 * @property Carbon $invoice_date
 * @property Carbon $sales_time
 * @property string|null $void_reason
 * @property string|null $invoice_type
 * @property Carbon|null $date_voided
 * @property Carbon|null $void_time
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 *
 * @property User|null $user
 * @property Customer|null $customer
 * @property Collection|InvoiceItemBatch[] $invoice_item_batches
 * @property Collection|InvoiceItem[] $invoice_items
 *
 * @package App\Models
 */
class Invoice extends Model
{
    use LogsActivity;

    protected $table = 'invoices';

    protected $casts = [
        'customer_id' => 'int',
        'payment_id' => 'int',
        'warehousestore_id' => 'int',
        'discount_amount' => 'float',
        'sub_total' => 'float',
        'total_amount_paid' => 'float',
        'total_profit' => 'float',
        'total_cost' => 'float',
        'vat' => 'float',
        'vat_amount' => 'float',
        'created_by' => 'int',
        'last_updated_by' => 'int',
        'voided_by' => 'int',
        'is_approve' => 'boolean'
    ];

    protected $dates = [
        'invoice_date',
        'sales_time',
        'date_voided',
        'void_time'
    ];

    protected $fillable = [
        'invoice_number',
        'warehousestore_id',
        'payment_id',
        'invoice_type',
        'invoice_paper_number',
        'customer_id',
        'discount_type',
        'discount_amount',
        'department',
        'status',
        'sub_total',
        'total_amount_paid',
        'total_profit',
        'total_cost',
        'vat',
        'vat_amount',
        'created_by',
        'last_updated_by',
        'voided_by',
        'invoice_date',
        'sales_time',
        'void_reason',
        'date_voided',
        'void_time',
        'is_approve'
    ];

    public function created_by()
    {
        return $this->belongsTo(User::class, 'created_by');
    }


    public function warehousestore()
    {
        return $this->belongsTo(Warehousestore::class);
    }

    public function created_user()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function last_updated()
    {
        return $this->belongsTo(User::class, 'last_updated_by');
    }

    public function user()
    {
        return $this->belongsTo(User::class, 'voided_by');
    }

    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    public function invoice_item_batches()
    {
        return $this->hasMany(InvoiceItemBatch::class);
    }

    public function invoice_items()
    {
        return $this->hasMany(InvoiceItem::class);
    }

    public function payment()
    {
        return $this->morphOne(Payment::class,'invoice');
    }

    public function paymentMethodTable()
    {

        return $this->morphMany(PaymentMethodTable::class,'invoice');
    }

    protected static function boot()
    {
        parent::boot(); // TODO: Change the autogenerated stub

        self::deleting(function ($invoice){

        });
    }


    public function approveInvoice()
    {
        $this->is_approve = 1;
        $this->status ="APPROVED";
        $this->update();
    }

    public function declineInvoice()
    {
        $this->is_approve = 0;
        $this->status ="DELETED";
        $this->update();

        foreach ($this->invoice_item_batches()->get() as $invoice_batch){
            $batch = $invoice_batch->stockbatch;
            $batch->{$invoice_batch->store} += $invoice_batch->quantity;
            $batch->update();
        }

    }


    public function destroyInvoice()
    {
        $this->declineInvoice();
        if($this->payment_id != NULL)
        {
            $id = $this->payment_id;
            $this->payment_id = NULL;
            $this->update();
            Payment::find($id)->delete();
        }
    }


    public static function updateInvoice($request, $reports, $invoice){
        if($invoice!=false){

            //delete invoice payment

            $invoice->invoice_items()->delete();

            $invoice->invoice_item_batches()->delete();

            $payment_id = $invoice->payment_id;

            $invoice->payment_id = NULL;

            $invoice->sales_representative_id = $request->get("sales_rep") == "0" ? NULL : $request->get("sales_rep");

            $invoice->total_amount_paid = 0;

            $invoice->update();

            $payment_info = [];

            if($payment_id != NULL) {
                $payment = Payment::find($payment_id);
                $payment_info = ['date'=>$payment->payment_date,'time'=>$payment->payment_time];
                $payment->delete();
            }




        }

        $invoice_paper_number = time();

        $totals = self::calculateInvoiceTotal($reports);

        //now lets update the invoice

        $invoice->invoice_paper_number = $invoice_paper_number;
        if(config('app.store') == "inventory")
        {
            $invoice->customer_id = $request->get('customer_id');
        }
        $invoice->status = $request->get('status');
        $invoice->sub_total = $totals['total_invoice_total_selling'];
        $invoice->total_amount_paid = 0;
        $invoice->total_profit = $totals['total_invoice_total_profit'];
        $invoice->total_cost = $totals['total_invoice_total_cost'];
        $invoice->last_updated_by = auth()->id();

        $invoice->update();


        //create invoice items

        $invoice_items_data = self::prepareInvoiceItemData(
            $reports,
            $request->get('customer_id'),
            $request->get('status'),
            $request->get('date'),
            Carbon::now()->toTimeString(),
            $invoice
        );

        $invoice_items_batches_data = self::prepareInvoiceItemBatchesData(
            $reports,
            $request->get('customer_id'),
            $request->get('status'),
            $request->get('date'),
            Carbon::now()->toTimeString(),
            $invoice
        );


        foreach ($invoice_items_data as $key=>$invoice_items_datum){
            $invoice
                ->invoice_items()
                ->save($invoice_items_datum)
                ->invoice_item_batches()
                ->saveMany($invoice_items_batches_data[$key]);
        }

        if(count($reports['returnLogs']) > 0)
            foreach ($reports['returnLogs'] as $log){
                $log['date_added'] = date('Y-m-d');
                ReturnLog::create($log);
            }

        session()->put("payment_info",$payment_info);

        return $invoice;

    }
    public static function validateInvoiceUpdateProduct($products,$store, $invoice)
    {

        $returnLogs = [];

        //check if a product was removed fro return invoice
        if($invoice->status == "COMPLETE"){

            $existing_invoice_items = $invoice->invoice_items()->pluck('id')->toArray();

            $new_invoice_items = Arr::pluck($products, 'invoice_item_id');

            $diff = array_diff($existing_invoice_items, $new_invoice_items);

            if (count($diff) > 0) {
                foreach (InvoiceItem::whereIn('id', $diff)->get() as $removedItem) {
                    $returnLogs[] = [
                        'store_after' => $removedItem->type,
                        'stock_id' => $removedItem->stock_id,
                        'warehousestore_id' => getActiveStore()->id,
                        'invoice_number' => $invoice->invoice_number,
                        'invoice_paper_number' => $invoice->invoice_paper_number,
                        'customer_id' => $invoice->customer_id,
                        'quantity_before' => $removedItem->quantity,
                        'quantity_after' => $removedItem->quantity,
                        'selling_price' => $removedItem->selling_price,
                        'date' => dailyDate(),
                        'dif' => $removedItem->quantity,
                        'store_before' => $removedItem->type,
                        'user_id' => auth()->id()
                    ];
                }
            }

            foreach ($products as $product) {

                if (isset($product['invoice_item_id']) && $product['invoice_item_id'] == "new") continue;

                $invoice_item = InvoiceItem::find($product['invoice_item_id']);

                if (!$invoice_item) continue;

                if (($product['qty'] < $invoice_item->quantity) || ($invoice_item->store != $product['type'])) {
                    $returnLogs[] = [
                        'store_after' => $product['type'],
                        'stock_id' => $product['id'],
                        'warehousestore_id' => getActiveStore()->id,
                        'invoice_number' => $invoice->invoice_number,
                        'invoice_paper_number' => $invoice->invoice_paper_number,
                        'customer_id' => $invoice->customer_id,
                        'quantity_before' => $invoice_item->quantity,
                        'quantity_after' => $product['qty'],
                        'selling_price' => $product['price'],
                        'date' => dailyDate(),
                        'dif' => ($invoice_item->quantity - $product['qty']),
                        'store_before' => $invoice_item->store,
                        'user_id' => auth()->id()
                    ];
                }

            }

        }

        //return product item batches
        foreach ($invoice->invoice_item_batches()->get() as $invoice_batch){
            $batch = $invoice_batch->stockbatch;
            $batch->{$invoice_batch->store} += $invoice_batch->quantity;
            $batch->update();
        }

        $status = false;

        $report = [];
        $errors = [];
        $prods = [];
        $below_cost_price = [];

        $product_ids =  array_column($products,'id');

        foreach ($products as $product){
            $prods[$product['id']] = $product;
        }

        $stocks = Stock::whereIn('id',$product_ids)->get();

        foreach ($stocks as $stock){
            $batches = $stock->getSaleableBatches($prods[$stock->id]['type'],$prods[$stock->id]['qty']);
            if($batches == false){
                $status = true;
                $errors[$stock->id] = "Not enough quantity to process ".$stock->name;
            }
            if($prods[$stock->id]['price'] <=  $stock->cost_price)
            {
                $below_cost_price[] = [
                    'name' => $stock->name,
                    'cost_price' =>$stock->cost_price,
                    'selling_price' => $prods[$stock->id]['price']
                ];
            }

            $report[$stock->id]['batches'] =  $batches;
            $report[$stock->id]['stock'] = $stock;
            $report[$stock->id]['prods'] = $prods[$stock->id];
        }

        if(count($errors) > 0){ // a validation error occurred return the batch back
            foreach ($invoice->invoice_item_batches()->get() as $invoice_batch){
                $batch = $invoice_batch->stockbatch;
                $batch->{$invoice_batch->store} -= $invoice_batch->quantity;
                $batch->update();
            }
        }

        return ['status'=> $status, 'data'=>$report,'errors'=> $errors,'returnLogs'=>$returnLogs,"below_cost_price"=>[]]; //$below_cost_price

    }

    public static function createInvoice($request, $reports){

        $totals = self::calculateInvoiceTotal($reports);

        $invoice_paper_number = time();

        $invoice_data = [
            'invoice_number'=>time(),
            'invoice_paper_number' => $invoice_paper_number,
            'customer_id'=> $request->get('customer_id'),
            'discount_type'=> "none",
            'discount_amount' => 0,
            'department'=> auth()->user()->department,
            'warehousestore_id' => getActiveStore()->id,
            'status'=>$request->get('status'),
            'sales_representative_id' => $request->get("sales_rep") == "0" ? NULL : $request->get("sales_rep"),
            'sub_total' => $totals['total_invoice_total_selling'],
            'total_amount_paid' => 0,
            'total_profit' =>  $totals['total_invoice_total_profit'],
            'total_cost' => $totals['total_invoice_total_cost'],
            'vat' => 0,
            'vat_amount' => 0,
            'created_by' => auth()->id(),
            'last_updated_by' =>auth()->id(),
            'invoice_date' =>  $request->get('date'),
            'sales_time' =>Carbon::now()->toTimeString(),
            'is_approve' => $request->get('status') == "REQUIRED-APPROVAL" ? 1 : 0
        ];

        $invoice = Invoice::create($invoice_data);


        //create invoice items

        $invoice_items_data = self::prepareInvoiceItemData(
            $reports,
            $request->get('customer_id'),
            $request->get('status'),
            $request->get('date'),
            Carbon::now()->toTimeString(),
            $invoice
        );

        $invoice_items_batches_data = self::prepareInvoiceItemBatchesData(
            $reports,
            $request->get('customer_id'),
            $request->get('status'),
            $request->get('date'),
            Carbon::now()->toTimeString(),
            $invoice
        );


        foreach ($invoice_items_data as $key=>$invoice_items_datum){
            $invoice
                ->invoice_items()
                ->save($invoice_items_datum)
                ->invoice_item_batches()
                ->saveMany($invoice_items_batches_data[$key]);
        }
        return $invoice;
    }

    public static function validateInvoiceProduct($products,$store){

        $status = false;
        $report = [];
        $errors = [];
        $prods = [];
        $below_cost_price = [];

        $product_ids =  array_column($products,'id');

        foreach ($products as $product){
            $prods[$product['id']] = $product;
        }

        $stocks = Stock::whereIn('id',$product_ids)->get();

        foreach ($stocks as $stock){
            $batches = $stock->getSaleableBatches($prods[$stock->id]['type'],$prods[$stock->id]['qty']);
            if($batches == false){
                $status = true;
                $errors[$stock->id] = "Not enough quantity to process ".$stock->name;
            }
            //echo $stock->cost_price ."<=". $prods[$stock->id]['price'];
            if($prods[$stock->id]['price'] <=  $stock->cost_price)
            {
                $below_cost_price[] = [
                    'name' => $stock->name,
                    'cost_price' =>$stock->cost_price,
                    'selling_price' => $prods[$stock->id]['price']
                ];
            }

            $report[$stock->id]['batches'] =  $batches;
            $report[$stock->id]['stock'] = $stock;
            $report[$stock->id]['prods'] = $prods[$stock->id];
        }

        return ['status'=> $status, 'data'=>$report,'errors'=> $errors, "below_cost_price"=>[]]; //$below_cost_price

    }


    public static function validateDepositPaymentApprovedPayNowUsage($invoice)
    {

        $deposit_used = 0;

        $paymentInformation =  (is_array(request()->get('payment')) ? request()->get('payment') : json_decode(request()->get('payment'),true));

        if(isset($paymentInformation['payment_method_id']) && $paymentInformation['payment_method_id'] == "split_method")
        {
            foreach($paymentInformation['split_method'] as $pmthod=>$amount) {

                if ($pmthod == 5 && $amount > 0) {
                    if ($invoice->customer_id == 1) return ['status' => true, 'errors' => "Please select a customer for payment with deposit"];

                    $customer = $invoice->customer;

                    if(!$customer)
                    {
                        return ['status'=>true,'errors'=>"Please select a valid customer for payment with deposit"];
                        break;
                    }

                    if(($customer->deposit_balance + $deposit_used) < $amount)
                    {
                        return ['status'=>true,'errors'=>"Insufficient deposit amount, please add more deposit to complete this transaction"];
                        break;
                    }

                }
            }

            DB::statement('SET FOREIGN_KEY_CHECKS=0;');
            Payment::where('invoice_id',$invoice->id)->where('invoice_type',CustomerDepositsHistory::class)->delete();
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
        }


        if(isset($paymentInformation['payment_method_id']) && $paymentInformation['payment_method_id'] == 5)
        {
            $totals = [
                'total_invoice_total_selling'=>$invoice->sub_total,
                'total_invoice_total_cost' =>$invoice->total_cost,
                'total_invoice_total_profit' =>$invoice->total_profit
            ];

            if($invoice->customer_id == "1")
            {
                return ['status'=>true,'errors'=>"Please select a customer for payment with deposit"];

            }

            if(($invoice->customer->deposit_balance + $deposit_used) <  $totals['total_invoice_total_selling'])
            {
                return ['status'=>true,'errors'=>"Insufficient deposit amount, please add more deposit to complete this transaction"];
            }

            DB::statement('SET FOREIGN_KEY_CHECKS=0;');
            Payment::where('invoice_id',$invoice->id)->where('invoice_type',CustomerDepositsHistory::class)->delete();
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');

        }

        return true;
    }

    public static function validateDepositPaymentUsage($validateInvoiceProductReport,$id=0)
    {
        $deposit_used = 0;

        if($id!=0)
        {

            $deposit__ = PaymentMethodTable::where('invoice_id',$id)->where('invoice_type',CustomerDepositsHistory::class)->get();

            if($deposit__->count() > 0)
            {
                $deposit_used = $deposit__->sum('amount');
            }

        }


        $paymentInformation =  (is_array(request()->get('payment')) ? request()->get('payment') : json_decode(request()->get('payment'),true));
        //request()->get('payment');
        if(isset($paymentInformation['payment_method_id']) && $paymentInformation['payment_method_id'] == "split_method")
        {
            foreach($paymentInformation['split_method'] as $pmthod=>$amount){

                if ($pmthod == 5 && $amount > 0)
                {
                    //check if the customer has enough deposit for the transaction

                    if(request()->get('customer_id') == "1")
                    {
                        return ['status'=>true,'errors'=>"Please select a customer for payment with deposit"];
                        break;
                    }

                    $customer = Customer::find(request()->get('customer_id'));

                    if(!$customer)
                    {
                        return ['status'=>true,'errors'=>"Please select a valid customer for payment with deposit"];
                        break;
                    }

                    if(($customer->deposit_balance + $deposit_used) < $amount)
                    {
                        return ['status'=>true,'errors'=>"Insufficient deposit amount, please add more deposit to complete this transaction"];
                        break;
                    }

                }

            }

            DB::statement('SET FOREIGN_KEY_CHECKS=0;');
            Payment::where('invoice_id',$id)->where('invoice_type',CustomerDepositsHistory::class)->delete();
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');

        }

        if(isset($paymentInformation['payment_method_id']) && $paymentInformation['payment_method_id'] == 5)
        {
            $totals = self::calculateInvoiceTotal($validateInvoiceProductReport);

            if(request()->get('customer_id') == "1")
            {
                return ['status'=>true,'errors'=>"Please select a customer for payment with deposit"];

            }

            $customer = Customer::find(request()->get('customer_id'));

            if(!$customer)
            {
                return ['status'=>true,'errors'=>"Please select a valid customer for payment with deposit"];
            }

            if(($customer->deposit_balance + $deposit_used) <  $totals['total_invoice_total_selling'])
            {
                return ['status'=>true,'errors'=>"Insufficient deposit amount, please add more deposit to complete this transaction"];
            }
            DB::statement('SET FOREIGN_KEY_CHECKS=0;');
            Payment::where('invoice_id',$id)->where('invoice_type',CustomerDepositsHistory::class)->delete();
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');

        }



        return $validateInvoiceProductReport;
    }


    public static function calculateInvoiceTotal($validationReports){

        $stocks = $validationReports['data'];

        $invoiceTotal = [];

        $total_invoice_total_selling = 0;
        $total_invoice_total_cost = 0;
        $total_invoice_total_profit = 0;

        foreach ($stocks as $key=>$stock){
            //lets check if the customer is buying yards or bundle
            if($stock['prods']['type'] == "yard_qty"){
                $stock['stock']->selling_price = $stock['prods']['price'];
                $stock['stock']->cost_price = $stock['stock']->yard_cost_price;
            }
            $invoiceTotal[$key]['total_selling_price']  =  $stock['prods']['price'] * $stock['prods']['qty'];
            $invoiceTotal[$key]['total_cost_price']  =  $stock['stock']->cost_price * $stock['prods']['qty'];
            $invoiceTotal[$key]['total_profit'] =  $invoiceTotal[$key]['total_selling_price'] -  $invoiceTotal[$key]['total_cost_price'];
            $total_invoice_total_selling += $invoiceTotal[$key]['total_selling_price'];
            $total_invoice_total_cost += $invoiceTotal[$key]['total_cost_price'] ;
            $total_invoice_total_profit+=$invoiceTotal[$key]['total_profit'];

        }

        return [
            'total_invoice_total_selling'=>$total_invoice_total_selling,
            'total_invoice_total_cost' =>$total_invoice_total_cost,
            'total_invoice_total_profit' =>$total_invoice_total_profit
        ];
    }


    public static function prepareInvoiceItemData($validationReports,$customer_id, $status, $invoice_date,$sales_time, $invoice)
    {

        $stocks = $validationReports['data'];

        $invoiceItems = [];

        $returnLogs = [];

        foreach ($stocks as $key=>$stock){

            if($stock['prods']['type'] == "yard_qty"){
                $stock['stock']->selling_price = $stock['prods']['price'];
                $stock['stock']->cost_price = $stock['stock']->yard_cost_price;
            }

            $total_selling_price  =  $stock['prods']['price'] * $stock['prods']['qty'];
            $total_cost_price  =  $stock['stock']->cost_price * $stock['prods']['qty'];
            $total_profit =   $total_selling_price -  $total_cost_price;

            //remove stock quantity from database

            $invoiceItems[$key] =  new InvoiceItem([
                'invoice_id'=> $invoice->id,
                'stock_id'=>$key,
                'quantity'=>$stock['prods']['qty'],
                'customer_id'=>$customer_id,
                'department'=> auth()->user()->department,
                'status' => $status,
                'added_by'=> auth()->id(),
                'warehousestore_id' => getActiveStore()->id,
                'invoice_date' =>$invoice_date,
                'store' => $stock['prods']['type'],
                'sales_time' =>$sales_time,
                'cost_price'=>($invoice->sub_total < 0 ? -($stock['stock']->cost_price) : ($stock['stock']->cost_price)),
                'selling_price' =>($invoice->sub_total < 0 ? -($stock['prods']['price']) : ($stock['prods']['price'])),
                'profit'=>($invoice->sub_total < 0 ? ($stock['prods']['price'] - $stock['stock']->cost_price) : ($stock['prods']['price'] - $stock['stock']->cost_price)),
                'total_selling_price' =>($invoice->sub_total < 0 ? -$total_selling_price : $total_selling_price),
                'total_cost_price' => ($invoice->sub_total < 0 ? -$total_cost_price : $total_cost_price),
                'total_profit'=>($invoice->sub_total < 0 ? -$total_profit : $total_profit),
                'discount_type'=>'none',
                'discount_amount'=>0,
            ]);


            if($invoice->sub_total < 0) {
                // create a return log of the invoice total is less than zero
                $returnLogs[] = [
                    'store_after' => $stock['prods']['type'],
                    'stock_id' => $stock['stock']->id,
                    'warehousestore_id' => getActiveStore()->id,
                    'invoice_number' => $invoice->invoice_number,
                    'invoice_paper_number' => $invoice->invoice_paper_number,
                    'customer_id' => $invoice->customer_id,
                    'quantity_before' => $stock['prods']['qty'],
                    'quantity_after' => $stock['prods']['qty'],
                    'selling_price' => $stock['prods']['price'],
                    'date' => dailyDate(),
                    'dif' => 0,
                    'store_before' =>  $stock['prods']['type'],
                    'user_id' => auth()->id(),
                    'date_added' => dailyDate()
                ];

            }

        }

        foreach ($returnLogs as $log){
            ReturnLog::create($log);
        }

        return $invoiceItems;
    }

    public static function prepareInvoiceItemBatchesData($validationReports,$customer_id, $status, $invoice_date,$sales_time, $invoice){

        $invoiceItemBatches = [];

        $stocks = $validationReports['data'];

        foreach ($stocks as $key=>$stock){
            if($stock['prods']['type'] == "yard_qty"){
                $stock['stock']->selling_price = $stock['prod']['price'];
                $stock['stock']->cost_price = $stock['stock']->yard_cost_price;
            }

            if($invoice->sub_total < 0)
            {
                $stock['stock']->addSaleableBatches($stock['batches'],$stock['prods']['type']);
            }
            else
            {
                $stock['stock']->removeSaleableBatches($stock['batches']);
            }



            foreach ($stock['batches'] as $batch) {

                $invoiceItemBatches[$key][] =  new InvoiceItemBatch( [
                    'invoice_id'=> $invoice->id,
                    'stock_id' => $key,
                    'stockbatch_id'=>$batch['id'],
                    'cost_price'=> $invoice->sub_total < 0 ? -($stock['stock']->cost_price) : $stock['stock']->cost_price,
                    'selling_price'=> $invoice->sub_total < 0 ? -($stock['prods']['price']) : $stock['stock']->cost_price,
                    'profit'=>$invoice->sub_total < 0 ? -($stock['stock']->selling_price - $stock['stock']->cost_price) : ($stock['stock']->selling_price - $stock['stock']->cost_price),
                    'quantity' => $batch['qty'],
                    'department'=> auth()->user()->department,
                    'warehousestore_id' => getActiveStore()->id,
                    'store' => $stock['prods']['type'],
                    'invoice_date' => $invoice_date,
                    'sales_time' =>$sales_time,
                    'customer_id' =>$customer_id,
                ]);
            }

        }

        return $invoiceItemBatches;

    }


    //for return Invoice update

    public static function validateReturnInvoiceProduct($products,$store,$request){
        /*
        $invoice_paper_number_count = Invoice::where('invoice_paper_number',$request->get('invoice_paper_number'))->count();

        if ($invoice_paper_number_count == 0)
            return  ['status'=>true,'error'=>true, 'singleError'=>"Invoice Number entered does not exist, please check invoice number and try again"];
        */

        $status = false;
        $report = [];
        $errors = [];
        $prods = [];

        $product_ids =  array_column($products,'id');

        foreach ($products as $product){
            $prods[$product['id']] = $product;
        }

        $stocks = Stock::whereIn('id',$product_ids)->get();

        foreach ($stocks as $stock){
            $batches = $stock->getRecentBatchesForReturn($prods[$stock->id]['type'],$prods[$stock->id]['qty']);
            if($batches == false){
                $status = true;
                $errors[$stock->id] = "Not enough quantity to process ".$stock->name;
            }
            $report[$stock->id]['batches'] =  $batches;
            $report[$stock->id]['stock'] = $stock;
            $report[$stock->id]['prods'] = $prods[$stock->id];
        }

        return ['status'=> $status, 'data'=>$report,'errors'=> $errors];

    }



    public static function ReturnInvoice($request, $reports){

        $totals = self::calculateInvoiceTotal($reports);

        $invoice_paper_number = $request->get('invoice_paper_number');

        $invoice_paper_number_count = Invoice::where('invoice_paper_number',$invoice_paper_number)->count();

        if ($invoice_paper_number_count > 0) $invoice_paper_number = $invoice_paper_number."-".mt_rand();


        $invoice_data = [
            'invoice_number'=>time(),
            'invoice_paper_number' => $invoice_paper_number,
            'customer_id'=> $request->get('customer_id'),
            'discount_type'=> "none",
            'discount_amount' => 0,
            'warehousestore_id' => getActiveStore()->id,
            'status'=>"COMPLETE",
            'sub_total' => -$totals['total_invoice_total_selling'],
            'total_amount_paid' => 0,
            'total_profit' =>  -$totals['total_invoice_total_profit'],
            'total_cost' => -$totals['total_invoice_total_cost'],
            'vat' => 0,
            'vat_amount' => 0,
            'created_by' => auth()->id(),
            'last_updated_by' =>auth()->id(),
            'invoice_date' =>  $request->get('date'),
            'sales_time' =>Carbon::now()->toTimeString(),
        ];

        $invoice = Invoice::create($invoice_data);

        //create invoice items

        $invoice_items_data = self::prepareInvoiceItemData(
            $reports,
            $request->get('customer_id'),
            $request->get('status'),
            $request->get('date'),
            Carbon::now()->toTimeString(),
            $invoice
        );

        $invoice_items_batches_data = self::prepareInvoiceItemBatchesData(
            $reports,
            $request->get('customer_id'),
            $request->get('status'),
            $request->get('date'),
            Carbon::now()->toTimeString(),
            $invoice
        );


        foreach ($invoice_items_data as $key=>$invoice_items_datum){
            $invoice
                ->invoice_items()
                ->save($invoice_items_datum)
                ->invoice_item_batches()
                ->saveMany($invoice_items_batches_data[$key]);
        }


        return $invoice;


    }


}
